#include "Window.h"
#include "MenuResource.h"
#include "ListBox.h"
#include "Database.h"
#include "Slashes.h"
#include "FileDialog.h"
#include "Font.h"
#include "RecordDialog.h"
#include "RecordDialogResource.h"
#include "ColorDialog.h"
#include "Brush.h"
#include "AcceleratorTable.h"
#include "Splitter.h"
#include "Settings.h"
#include <commctrl.h>
#include <Shellapi.h>
#include <tchar.h>

using namespace std;
using namespace CWindows ;

class Application {
public:
  static const int WINDOW_STYLE = WS_OVERLAPPEDWINDOW|WS_CLIPCHILDREN;
  static const TCHAR WINDOW_TITLE[] ;
  static const TCHAR ABOUT_TITLE[] ;
  static const TCHAR ABOUT_TEXT[] ;
  static const int ID_SELECT_ALL = 20000 ;
public:
  Application();
  ~Application();
  int run( HINSTANCE hInst, HINSTANCE hPrev, LPSTR args, int nShow );
private:
  void drawInsertLine( HWND hwndListBox, int index );
  void displayRecordsPopupMenu(int screenX, int screenY);
  HMENU getRecordsMenu();
  void initAcceleratorTable();
  void onAbout() ;
  void onAdd() ;
  void onBackgroundColor() ;
  LRESULT onClose(WPARAM, LPARAM);
  bool loadFile( const String& );
  LRESULT onBeginDrag( LPDRAGLISTINFO );
  void onColor() ;
  LRESULT onCommand( WPARAM, LPARAM );
  LRESULT onCtlColorListBox( WPARAM, LPARAM );
  LRESULT onDragging( LPDRAGLISTINFO );
  LRESULT onDragListMessage( WPARAM, LPARAM );
  LRESULT onDestroy( WPARAM, LPARAM );
  LRESULT onDropFiles( WPARAM, LPARAM );
  LRESULT onDropped( LPDRAGLISTINFO );
  void onExit() ;
  void onFont() ;
  LRESULT onInitMenuPopup( WPARAM wParam, LPARAM lParam );
  void onItemSelectionChange();
  LRESULT onLButtonDown( WPARAM, LPARAM );
  LRESULT onLButtonUp( WPARAM, LPARAM );
  void onListBoxRightClick();
  LRESULT onMessage( HWND, UINT, WPARAM, LPARAM );
  LRESULT onMouseMove( WPARAM, LPARAM );
  LRESULT onMove( WPARAM, LPARAM );
  void onNew() ;
  void onOpen() ;
  void onRemove() ;
  void onRemoveAll() ;
  void onSave() ;
  void onSaveAs() ;
  void onSelectAll() ;
  LRESULT onSize( WPARAM wParam, LPARAM lParam );
  void onSortAscending() ;
  void onSortDescending() ;
  bool showConfirmBox( LPCTSTR qn ) const ;
  void updateLastSelectedRecord();
  void updateRecordDialog();
  void updateSelectedRecordName();
private:
  Splitter         splitter ;
  UINT             lastSelectedIndex ;
  Window           window ;
  ListBox          listBox ;
  String           fileName ;
  Database         database ;
  Font             font ;
  RecordDialog*    recordDialog ;
  Brush            textBackgroundBrush ;
  ColorDialog      textColorDialog ;
  ColorDialog      textBackgroundColorDialog ;
  AcceleratorTable acceleratorTable ;
  UINT             dragListMessage ;
  Settings         settings ;
  HMENU            recordsMenu ;
  UINT             listItemDraggingIndex ;
};

//------------------------------------------------------------------------------
// Application Static Data
//------------------------------------------------------------------------------

const TCHAR Application::WINDOW_TITLE[] = TEXT("Record Book 0.6 - Untitled");
const TCHAR Application::ABOUT_TITLE[] = TEXT("About Record Book");
const TCHAR Application::ABOUT_TEXT[] = TEXT(
  "Record Book - A Simple Text Database Tool\r\n"
  "Version 0.6\r\n"
  "Copyright 2006 Andrew Lim\r\n"
  "danteshamest@gmail.com\r\n\r\n"
  "Created with Dev-C++, UPX and MinGW\r\n"
);

//------------------------------------------------------------------------------
// Application Constructors & Destructors
//------------------------------------------------------------------------------

Application::Application()
  : lastSelectedIndex( (UINT)-1 )
  , font( "Courier New", 10 )
  , recordDialog(NULL)
  , dragListMessage(0)
  , recordsMenu(NULL)
  , listItemDraggingIndex( (UINT) -1) {
  initAcceleratorTable();

  // Window
  window.setMessageHandler<Application, &Application::onMessage>( *this );
  window.create( WINDOW_TITLE, NULL, WINDOW_STYLE, WS_EX_ACCEPTFILES );
  window.setClientSize( 400, 300 );
  window.centerInScreen() ;

  // If the menu has trouble being created, it's probably a resource compilation
  // error. Don't give identifiers to popup menus, MinGW doesn't seem to like
  // it.
  HMENU menuBar = LoadMenu(GetModuleHandle(NULL),MAKEINTRESOURCE(ID_MENUBAR)) ;
  if ( menuBar == NULL ) {
    MessageBox( window.getHandle(), TEXT("Error creating menu bar!"), 0, 0 ) ;
  }
  else {
    SetMenu( window.getHandle(), menuBar );
  }

  // Colors
  settings.load( "settings.txt" );
  const COLORREF color   = settings.getColor() ;
  const COLORREF bkcolor = settings.getBackgroundColor();
  textBackgroundBrush.createSolid( bkcolor );
  textColorDialog.setColor( color );
  textBackgroundColorDialog.setColor( bkcolor );

  // Font
  font = settings.getFont() ;
  
  // ListBox
  listBox.create( window.getHandle() ) ;
  listBox.makeDragList();
  listBox.setHorizontalExtent( 800 );
  font.setWindowFont( listBox, FALSE );
  dragListMessage = RegisterWindowMessage(DRAGLISTMSGSTRING);

  // Record Dialog
  recordDialog = new RecordDialog( window.getHandle() );
  recordDialog->setFont( font );
  recordDialog->setTextColor( color );
  recordDialog->setTextBackgroundColor( bkcolor );
  updateRecordDialog();
  recordDialog->setVisible( true );
  
  // Get handle of 'Records' menu.
  recordsMenu = getRecordsMenu();

  // Splitter Control
  splitter.x          = settings.getSplitterX() ;
  splitter.parent     = window.getHandle();
  splitter.leftChild  = listBox ;
  splitter.rightChild = *recordDialog ;
  
  bool windowMaximized = settings.isWindowMaximized() ;

  // Window size
  SetWindowPos( window.getHandle(), NULL, settings.getWindowX(),
                settings.getWindowY(), settings.getWindowWidth(),
                settings.getWindowHeight(), SWP_NOZORDER );
  if ( windowMaximized ) {
    ShowWindow( window.getHandle(), SW_SHOWMAXIMIZED );
  }
  else {
    ShowWindow( window.getHandle(), SW_SHOW );
  }
}

Application::~Application() {
}

//------------------------------------------------------------------------------
// Application Member Functions
//------------------------------------------------------------------------------

void Application::drawInsertLine( HWND hwndListBox, int index ) {
  RECT rc ;
  HDC hdc ;
  SendMessage( hwndListBox, LB_GETITEMRECT, index, (LPARAM)&rc );
  hdc = GetDC( hwndListBox );
  MoveToEx( hdc, rc.left, rc.top, NULL );
  LineTo( hdc, rc.right, rc.top );
  ReleaseDC( hwndListBox, hdc );
}

void Application::displayRecordsPopupMenu( int screenX, int screenY ) {
  TrackPopupMenu( recordsMenu, TPM_LEFTALIGN|TPM_TOPALIGN, screenX, screenY, 0,
                  window.getHandle(), 0 );
}

HMENU Application::getRecordsMenu() {
  return GetSubMenu( GetMenu(window.getHandle()), 1 );
}

void Application::initAcceleratorTable() {
  static ACCEL accels[] = {
      { FCONTROL|FVIRTKEY, 'S', ID_SAVE }       // Ctrl+S
    , { FCONTROL|FVIRTKEY, 'O', ID_OPEN }       // Ctrl+O
    , { FCONTROL|FVIRTKEY, 'N', ID_NEW }        // Ctrl+N
    , { FCONTROL|FVIRTKEY, 'A', ID_SELECT_ALL } // Ctrl+A
    , { FCONTROL|FVIRTKEY, 0x000000bb, ID_ADD } // Ctrl++
    , { FCONTROL|FVIRTKEY, 0x0000006b, ID_ADD } // Ctrl++
  };
  
  const int ACCEL_COUNT = sizeof( accels ) / sizeof( accels[0] ) ;
  if ( !acceleratorTable.create( accels, ACCEL_COUNT ) ) {
    MessageBox( window.getHandle(), TEXT("Error creating accelerator table!"),
                0, 0 );
  }
}

void Application::onAbout() {
  MessageBox( window.getHandle(), ABOUT_TEXT, ABOUT_TITLE, MB_ICONINFORMATION);
}

void Application::onAdd() {
  updateLastSelectedRecord();
  Record newRecord( "New Record" );
  database.add( newRecord );
  listBox.add( "New Record" );
  listBox.setSelectedIndex( (UINT)(database.getSize() - 1) );
  updateRecordDialog();
  
  // Give the "Name" textbox the keyboard focus, and select all text within it.
  const HWND nameBox = recordDialog->getNameBox();
  SetFocus( nameBox );
  SendMessage( nameBox, EM_SETSEL, 0, -1 );
}

void Application::onBackgroundColor() {
  if ( textBackgroundColorDialog.show( window.getHandle() ) ) {
    const COLORREF bkcolor = textBackgroundColorDialog.getColor();
    textBackgroundBrush.createSolid( bkcolor );
    recordDialog->setTextBackgroundColor( bkcolor );
    InvalidateRect( listBox, NULL, TRUE );
    InvalidateRect( *recordDialog, NULL, TRUE );
  }
}

LRESULT Application::onBeginDrag( LPDRAGLISTINFO info ) {
  // Get item index from cursor
  UINT nItem = LBItemFromPt( listBox, info->ptCursor, TRUE );
  // if index is valid
  if ( nItem >= 0 && nItem < database.getSize() ) {
    // update previous selected record
    updateLastSelectedRecord();
    // make sure new item is selected
    if ( nItem != listBox.getSelectedIndex() ) {
      listBox.setSelectedIndex( nItem );
      // display new item's name/value text
      updateRecordDialog();
    }
    // save the index of the item being dragged
    listItemDraggingIndex = nItem ;
    // return TRUE to allow operation
    return TRUE ;
  }
  // return FALSE to prevent operation
  return FALSE ;
}

LRESULT Application::onClose( WPARAM wParam, LPARAM lParam ) {
  onExit() ;
  return 0 ;
}

bool Application::loadFile( const String& fileToLoad ) {
  if ( database.load( fileToLoad ) ) {
    fileName = fileToLoad ;
    String title = TEXT("Record Book 0.6 - ") + fileName ;
    SetWindowText( window.getHandle(), title.c_str() );
    listBox.removeAll() ;
    for ( size_t i = 0; i < database.getSize(); ++i ) {
      Record& record = database.get( i ) ;
      const String& name = record.getName() ;
      listBox.add( name.c_str() );
    }
    updateRecordDialog();
    return true ;
  }
  return false ;
}

void Application::onColor() {
  if ( textColorDialog.show( window.getHandle() ) ) {
    recordDialog->setTextColor( textColorDialog.getColor() );
    InvalidateRect( listBox, NULL, FALSE );
    InvalidateRect( *recordDialog, NULL, FALSE );
  }
}

LRESULT Application::onCommand( WPARAM wParam, LPARAM lParam ) {
  WORD id   = LOWORD( wParam ) ;
  WORD code = HIWORD( wParam ) ;
  HWND ctrl = (HWND) lParam ;
  if ( ctrl == listBox ) {
    switch ( code ) {
      case LBN_SELCHANGE: onItemSelectionChange();   return 0 ;
      case LBN_SELCANCEL: onItemSelectionChange();   return 0 ;
      default:                                       return 0 ;
    }
  }
  switch ( id ) {
    case ID_NEW:             onNew()          ;    return 0 ;
    case ID_OPEN:            onOpen()         ;    return 0 ;
    case ID_SAVE:            onSave()         ;    return 0 ;
    case ID_SAVEAS:          onSaveAs()       ;    return 0 ;
    case ID_EXIT:            onExit()         ;    return 0 ;
    case ID_ADD:             onAdd()          ;    return 0 ;
    case ID_REMOVE:          onRemove()       ;    return 0 ;
    case ID_REMOVEALL:       onRemoveAll()    ;    return 0 ;
    case ID_SORTASC:         onSortAscending();    return 0 ;
    case ID_SORTDES:         onSortDescending();   return 0 ;
    case ID_FONT:            onFont()         ;    return 0 ;
    case ID_COLOR:           onColor()        ;    return 0 ;
    case ID_BACKGROUNDCOLOR: onBackgroundColor() ; return 0 ;
    case ID_ABOUT:           onAbout()        ;    return 0 ;
    case ID_SELECT_ALL:      onSelectAll();        return 0 ;
    default:                                       return 0 ;
  }
}

LRESULT Application::onCtlColorListBox( WPARAM wParam, LPARAM lParam ) {
  HDC hdc = (HDC) wParam ;
  SetTextColor( hdc, textColorDialog.getColor() );
  SetBkColor( hdc, textBackgroundColorDialog.getColor() );
  return (LRESULT) textBackgroundBrush.getHandle() ;
}

LRESULT Application::onDragging( LPDRAGLISTINFO info ) {
  // Holds previous insertion line.
  static int iOld = -1 ;
  // Get item index nearest to mouse.
  int nItem = LBItemFromPt( listBox, info->ptCursor, TRUE );
  // If item index is valid, draw an insertion line.
  if ( nItem != -1 ) {
    drawInsertLine( listBox, nItem );
  }
  // Erase previous insert line.
  if ( nItem != iOld && iOld != -1 ) {
    RECT rcOld ;
    listBox.getItemRect( iOld, &rcOld );
    InvalidateRect( listBox, &rcOld, TRUE );
  }
  // Current item index becomes old item index.
  iOld = nItem ;
  // Show copying cursor.
  return DL_COPYCURSOR ;
}

LRESULT Application::onDragListMessage( WPARAM wParam, LPARAM lParam ) {
  LPDRAGLISTINFO info = (LPDRAGLISTINFO) lParam ;
  switch (info->uNotification) {
    case DL_BEGINDRAG: return onBeginDrag( info ) ;
    case DL_DRAGGING:  return onDragging( info );
    case DL_DROPPED:   return onDropped( info );
    default:           return 0 ;
  }
}

LRESULT Application::onDestroy( WPARAM wParam, LPARAM lParam ) {
  PostQuitMessage( 0 ) ;
  return 0 ;
}

LRESULT Application::onDropFiles( WPARAM wParam, LPARAM lParam ) {
  HDROP hDrop = (HDROP) wParam ;
  TCHAR filePath[ MAX_PATH ];
  if ( DragQueryFile( hDrop, 0, filePath, MAX_PATH ) ) {
    loadFile( filePath );
  }
  return 0 ;
}

// Handles the DL_DROPPED notification.
LRESULT Application::onDropped( LPDRAGLISTINFO info ) {
  // Current and drop positions.
  UINT iCurr = listItemDraggingIndex ;
  UINT iDrop = LBItemFromPt( listBox, info->ptCursor, TRUE );
  // If drop position is valid and is not the list item's current position
  if ( iDrop >= 0 && iDrop < database.getSize() &&
       iCurr >= 0 && iCurr < database.getSize() &&
       iDrop != iCurr && iDrop != iCurr+1 ) {
    // Get element that was dragged.
    Record elem = database.get( iCurr );
    // Remove dragged element from its current position.
    database.remove( iCurr );
    listBox.remove( iCurr );
    // Insert dragged element into new position.
    UINT iInsert = iDrop > iCurr ? iDrop - 1 : iDrop ;
    database.insert( elem, iInsert );
    listBox.insert( elem.getName().c_str(), iInsert );
    // Select element at new position.
    listBox.setSelectedIndex( lastSelectedIndex = iInsert );
    // don't track drag item anymore
    listItemDraggingIndex = (UINT)-1 ;
  }
  return 0 ;
}

void Application::onExit() {
  if ( showConfirmBox( TEXT("Are you sure you wish to exit?") ) ) {
    settings.setColor( textColorDialog.getColor() );
    settings.setBackgroundColor( textBackgroundColorDialog.getColor() );
    settings.setFont( font );
    settings.setSplitterX( splitter.x );
    settings.save( "settings.txt" );
    PostQuitMessage( 0 ) ;
  }
}

void Application::onFont() {
  if ( font.showFontDialog( window.getHandle() ) ) {
    font.setWindowFont( listBox, TRUE );
    recordDialog->setFont( font );
  }
}

LRESULT Application::onInitMenuPopup( WPARAM wParam, LPARAM lParam ) {
  HMENU hMenu = GetMenu( window.getHandle() );
  bool selected = listBox.getSelectedIndex() != (UINT) -1 ;
  EnableMenuItem( hMenu, ID_UPDATE, selected?MF_ENABLED:MF_GRAYED) ;
  EnableMenuItem( hMenu, ID_REMOVE, selected?MF_ENABLED:MF_GRAYED) ;
  bool exists = database.getSize() > 0 ;
  EnableMenuItem( hMenu, ID_REMOVEALL, exists?MF_ENABLED:MF_GRAYED) ;
  EnableMenuItem( hMenu, ID_SORTASC, exists?MF_ENABLED:MF_GRAYED) ;
  EnableMenuItem( hMenu, ID_SORTDES, exists?MF_ENABLED:MF_GRAYED) ;
  /*
  CheckMenuItem( GetMenu(window.getHandle()),
                 ID_SOLUTION,
                 viewSolutionEnabled ? MF_CHECKED : MF_UNCHECKED );
  */
  return 0 ;
}

void Application::onItemSelectionChange() {
  if ( lastSelectedIndex != listBox.getSelectedIndex() ) {
    updateLastSelectedRecord();
    updateRecordDialog();
  }
}

LRESULT Application::onLButtonDown( WPARAM wParam, LPARAM lParam ) {
  splitter.onParentLButtonDown( wParam, lParam );
  return 0;
}

LRESULT Application::onLButtonUp( WPARAM wParam, LPARAM lParam ) {
  splitter.onParentLButtonUp( wParam, lParam );
  return 0;
}

void Application::onListBoxRightClick() {
  POINT cursorPos ;
  GetCursorPos( &cursorPos );
  UINT nItem = LBItemFromPt( listBox, cursorPos, FALSE );
  // if right-clicked an item, select that item and display its contents
  if ( nItem != (UINT)-1) {
    updateLastSelectedRecord();
    listBox.setSelectedIndex( (UINT)nItem );
    updateRecordDialog();
  }
  // display the popup menu
  displayRecordsPopupMenu( cursorPos.x, cursorPos.y );
}

LRESULT Application::onMessage( HWND hwnd,
                                UINT msg,
                                WPARAM wParam,
                                LPARAM lParam ) {
  if ( msg == dragListMessage ) {
    return onDragListMessage( wParam, lParam );
  }
  switch ( msg ) {
    case WM_CLOSE:           return onClose( wParam, lParam );
    case WM_COMMAND:         return onCommand( wParam, lParam );
    case WM_CTLCOLORLISTBOX: return onCtlColorListBox( wParam, lParam );
    case WM_DESTROY:         return onDestroy( wParam, lParam );
    case WM_DROPFILES:       return onDropFiles( wParam, lParam );
    case WM_INITMENUPOPUP:   return onInitMenuPopup( wParam, lParam );
    case WM_LBUTTONDOWN:     return onLButtonDown( wParam, lParam );
    case WM_LBUTTONUP:       return onLButtonUp( wParam, lParam );
    case WM_MOUSEMOVE:       return onMouseMove( wParam, lParam );
    case WM_MOVE:            return onMove( wParam, lParam );
    case WM_SIZE:            return onSize( wParam, lParam );
    default:                 return DefWindowProc( hwnd, msg, wParam, lParam );
  }
}

LRESULT Application::onMouseMove( WPARAM wParam, LPARAM lParam ) {
  splitter.onParentMouseMove( wParam, lParam );
  return 0;
}

LRESULT Application::onMove( WPARAM wParam, LPARAM lParam ) {
  const HWND hwnd = window.getHandle() ;
  if ( !IsZoomed(hwnd) && !IsIconic(hwnd) ) {
    RECT rcWindow ;
    GetWindowRect( window.getHandle(), &rcWindow );
    int windowX = rcWindow.left ;
    int windowY = rcWindow.top ;
    settings.setWindowX( windowX ) ;
    settings.setWindowY( windowY ) ;
  }
  return 0;
}

void Application::onNew() {
  if ( showConfirmBox("Are you sure you wish to start a new project?") ) {
    fileName.clear() ;
    SetWindowText( window.getHandle(), TEXT("Record Book 0.6 - Untitled") );
    database.removeAll() ;
    listBox.removeAll() ;
    updateRecordDialog();
  }
}

void Application::onOpen() {
  OpenFileDialog dialog ;
  if ( dialog.ShowDialog( window.getHandle() ) ) {
    if ( !loadFile( dialog.FileName() ) ) {
      MessageBox(window.getHandle(), "Error loading database file!", 0, 0);
    }
  }
}

void Application::onRemove() {
  UINT selectedIndex = listBox.getSelectedIndex() ;
  if ( selectedIndex != (UINT)-1 ) {
    if ( showConfirmBox("Remove the selected record?") ) {
      listBox.remove( selectedIndex );
      database.remove( selectedIndex );
      updateRecordDialog();
    }
  }
}

void Application::onRemoveAll() {
  // Confirm
  if (showConfirmBox( "Are you sure you wish to remove all records?" ) ) {
    listBox.removeAll() ;
    database.removeAll() ;
    updateRecordDialog();
  }
}

void Application::onSave() {
  updateLastSelectedRecord();
  if ( !fileName.empty() ) {
    if ( !database.save( fileName ) ) {
      MessageBox(window.getHandle(), "Error saving file!", 0, 0);
    }
  }
  else {
    onSaveAs() ;
  }
}

void Application::onSaveAs() {
  updateLastSelectedRecord();
  SaveFileDialog dialog ;
  if ( dialog.ShowDialog( window.getHandle() ) ) {
    if ( database.save( dialog.FileName() ) ) {
      fileName = dialog.FileName() ;
      const String title = TEXT("Record Book 0.6 - ") + fileName ;
      SetWindowText( window.getHandle(), title.c_str() );
    }
    else {
      MessageBox(window.getHandle(), "Error saving file!", 0, 0);
    }
  }
}

void Application::onSelectAll() {
  HWND hwndFocus = GetFocus();
  if ( hwndFocus == recordDialog->getNameBox() ||
       hwndFocus == recordDialog->getTextBox() ) {
    SendMessage( hwndFocus, EM_SETSEL, 0, -1 );
  }
}

LRESULT Application::onSize( WPARAM wParam, LPARAM lParam ) {
  splitter.onParentSize( wParam, lParam );
  const int resizeFlag = (int) wParam ;
  //const int clientW = LOWORD( lParam );
  //const int clientH = HIWORD( lParam );
  if ( resizeFlag == SIZE_MAXIMIZED ) {
    settings.setWindowMaximized( true );
  }
  else if ( resizeFlag == SIZE_RESTORED ) {
    settings.setWindowMaximized( false );
    RECT rcWindow ;
    GetWindowRect( window.getHandle(), &rcWindow );
    int windowX = rcWindow.left ;
    int windowY = rcWindow.top ;
    int windowW = rcWindow.right - rcWindow.left ;
    int windowH = rcWindow.bottom - rcWindow.top ;
    settings.setWindowX( windowX ) ;
    settings.setWindowY( windowY ) ;
    settings.setWindowWidth( windowW ) ;
    settings.setWindowHeight( windowH ) ;
  }
  return 0 ;
}

void Application::onSortAscending() {
  updateLastSelectedRecord();
  database.sortAscending() ;
  listBox.removeAll() ;
  for ( size_t i = 0; i < database.getSize(); ++i ) {
    const Record& record = database.get( i ) ;
    const String& name = record.getName() ;
    listBox.add( name.c_str() );
  }
  updateRecordDialog();
}

void Application::onSortDescending() {
  updateLastSelectedRecord();
  database.sortDescending() ;
  listBox.removeAll() ;
  for ( size_t i = 0; i < database.getSize(); ++i ) {
    const Record& record = database.get( i ) ;
    const String& name = record.getName() ;
    listBox.add( name.c_str() );
  }
  updateRecordDialog();
}

bool Application::showConfirmBox( LPCTSTR qn ) const {
  return MessageBox( window.getHandle(), qn, TEXT("Confirm"),
                     MB_YESNO|MB_DEFBUTTON2 ) == IDYES;
}

/**
 * This function saves the current user input into the last selected record.
 * It also updates the name of the last selected item in the list box.
 */
void Application::updateLastSelectedRecord() {
  if ( lastSelectedIndex != (UINT) -1 ) {
    UINT currentSelectedIndex = listBox.getSelectedIndex() ;
    Record& record = database.get( lastSelectedIndex );
    String name ; record.setName( recordDialog->getName(name) );
    String text ; record.setText( recordDialog->getText(text) );
    listBox.replace( lastSelectedIndex, name.c_str() );
    if ( currentSelectedIndex == lastSelectedIndex ) {
      listBox.setSelectedIndex( currentSelectedIndex );
    }
  }
}

// Updates the displayed contents of the record dialog depending on the current
// list item selection.
void Application::updateRecordDialog() {
  UINT selectedIndex = listBox.getSelectedIndex() ;
  if ( selectedIndex != (UINT) -1 ) {
    const Record& selectedRecord = database.get( selectedIndex );
    recordDialog->setName( selectedRecord.getName() );
    recordDialog->setText( selectedRecord.getText() );
    EnableWindow( recordDialog->getNameBox(), TRUE );
    EnableWindow( recordDialog->getTextBox(), TRUE );
  }
  else {
    recordDialog->setName( "" );
    recordDialog->setText( "" );
    EnableWindow( recordDialog->getNameBox(), FALSE );
    EnableWindow( recordDialog->getTextBox(), FALSE );
  }
  lastSelectedIndex = selectedIndex ;
}

void Application::updateSelectedRecordName() {
  UINT iSelected = listBox.getSelectedIndex() ;
  if ( iSelected != (UINT) -1 ) {
    Record& record = database.get( iSelected );
    const String& name = record.getName() ;
    String newName ;
    recordDialog->getName(newName) ;
    if ( name != newName ) {
      record.setName( newName );
      listBox.replace( iSelected, newName.c_str() );
      listBox.setSelectedIndex( iSelected );
    }
  }
}

int Application::run(HINSTANCE hInst, HINSTANCE hPrev, LPSTR args, int nShow ){
  // If there is a command line argument, treat it as a file name, and try to
  // load it.
  if ( __argc > 1 && _tcslen(__argv[1]) ) {
    if ( !loadFile( __argv[1] ) ) {
      MessageBox( window.getHandle(), __argv[1], 0, 0 );
    }
  }

  MSG msg ;
  while( GetMessage(&msg,0,0,0) > 0 ) {
    // Tab pressed in value box
    if ( msg.message == WM_KEYDOWN && msg.hwnd == recordDialog->getTextBox() &&
         msg.wParam == VK_TAB && !(GetAsyncKeyState(VK_CONTROL)&0x8000) ) {
      SetFocus( listBox );
    }
    // Del pressed in list box
    else if ( msg.message == WM_KEYDOWN && msg.hwnd == listBox &&
              msg.wParam == VK_DELETE ) {
      onRemove();
    }
    // Right-click list box
    else if ( msg.message == WM_RBUTTONDOWN && msg.hwnd == listBox ) {
      onListBoxRightClick();
    }
    else {
      if ( !acceleratorTable.translate(window.getHandle(),&msg) ) {
        if ( !IsDialogMessage(window.getHandle(),&msg) ) {
          TranslateMessage(&msg);
          DispatchMessage(&msg);
        }
        // START OF HACK
        // When it's possible the value in the "Name" edit control changes...
        if ( msg.hwnd    == recordDialog->getNameBox() &&
             GetFocus()  == recordDialog->getNameBox() &&
             msg.message != WM_MOUSEMOVE ) {
          // Update the item name in the list box.
          updateSelectedRecordName();
        }
        // END OF HACK
      }
    }
  }
  return (int)msg.wParam ;
}

int WINAPI WinMain( HINSTANCE hInst, HINSTANCE hPrev, LPSTR args, int nShow ) {
  InitCommonControls() ;
  Application application ;
  return application.run( hInst, hPrev, args, nShow );
}

