#include "RecordDialog.h"
#include "RecordDialogResource.h"

//------------------------------------------------------------------------------
// RecordDialog Static Data
//------------------------------------------------------------------------------

const TCHAR RecordDialog::PROPERTY_NAME[] = TEXT("A");
const TCHAR RecordDialog::ERROR_NO_NAME[] = TEXT("Enter the record name.") ;
const TCHAR RecordDialog::ERROR_NO_TEXT[] = TEXT("Enter the record value.") ;

//------------------------------------------------------------------------------
// RecordDialog Constructors & Destructors
//------------------------------------------------------------------------------

RecordDialog::RecordDialog( HWND hwndParent ) throw (CreationException)
  : hwnd( NULL )
  , textBox( NULL )
  , leftDiff( 0 )
  , topDiff( 0 )
  , rightDiff( 0 )
  , bottomDiff( 0 )
  , color(RGB(0,0,0)) {
  backgroundBrush.createSolid( RGB(255,255,255) ) ;
  if ( create( hwndParent ) ) {
    throw CreationException() ;
  }
}

RecordDialog::~RecordDialog() {
  destroy() ;
}

//------------------------------------------------------------------------------
// RecordDialog Public Member Functions
//------------------------------------------------------------------------------

bool RecordDialog::create( HWND hwndOwner ) {
  CreateDialogParam( GetModuleHandle( NULL ),
                     MAKEINTRESOURCE( ID_RECORDDIALOG ),
                     hwndOwner,
                     RecordDialog::DialogProc,
                     (LPARAM) this ) ;
  return hwnd == NULL ;
}

void RecordDialog::destroy() {
  if ( hwnd ) {
    DestroyWindow( hwnd ) ;
    hwnd = NULL ;
  }
}

String& RecordDialog::getName( String& name ) const {
  return getWindowText( nameBox, name );
}

HWND RecordDialog::getNameBox() const {
  return nameBox ;
}

String& RecordDialog::getText( String& text ) const {
  return getWindowText( textBox, text );
}

HWND RecordDialog::getTextBox() const {
  return textBox ;
}

COLORREF RecordDialog::getTextBackgroundColor() const {
  return backgroundBrush.getColor() ;
}

COLORREF RecordDialog::getTextColor() const {
  return color ;
}

bool RecordDialog::isDialogMessage( LPMSG lpmsg ) const {
  return IsDialogMessage( hwnd, lpmsg ) != 0 ;
}

bool RecordDialog::isVisible() const {
  return IsWindowVisible( hwnd ) != 0 ;
}

void RecordDialog::setName( const String& newName ) {
  SetWindowText( nameBox, newName.c_str() );
}

void RecordDialog::setText( const String& newText ) {
  SetWindowText( textBox, newText.c_str() );
}

void RecordDialog::setTextBackgroundColor( COLORREF newBackgroundColor ) {
  backgroundBrush.createSolid( newBackgroundColor );
  InvalidateRect( hwnd, NULL, FALSE );
}
void RecordDialog::setTextColor( COLORREF newColor ) {
  color = newColor ;
  InvalidateRect( hwnd, NULL, FALSE );
}

void RecordDialog::setFont( const Font& f ) {
  font = f ;
  font.setWindowFont( nameBox, true );
  font.setWindowFont( textBox, true );
}

void RecordDialog::setVisible( bool b ) {
  ShowWindow( hwnd, b ? SW_SHOW : SW_HIDE );
}

//------------------------------------------------------------------------------
// RecordDialog Private Member Functions
//------------------------------------------------------------------------------

INT_PTR CALLBACK RecordDialog::DialogProc( HWND   hwnd,
                                           UINT   msg,
                                           WPARAM wParam,
                                           LPARAM lParam ) {
  RecordDialog* recordDialog = (RecordDialog*) GetProp( hwnd, PROPERTY_NAME );
  if ( recordDialog != NULL ) {
    return recordDialog->onMessage( hwnd, msg, wParam, lParam );
  }
  else if ( msg == WM_INITDIALOG ) {
    // Get pointer-to-RecordDialog passed in DialogBoxParam.
    RecordDialog* recordDialog = reinterpret_cast<RecordDialog*>(lParam);

    // Make pointer-to-RecordDialog a window property of the dialog box.
    SetProp( (recordDialog->hwnd = hwnd), PROPERTY_NAME, recordDialog );

    // Initialize the text boxes.
    recordDialog->nameBox = GetDlgItem( hwnd, ID_NAMEBOX ) ;
    recordDialog->textBox = GetDlgItem( hwnd, ID_RECORD_TEXTBOX ) ;
    
    // Set the input limit of the text boxes to the max
    SendMessage( recordDialog->nameBox, EM_LIMITTEXT, 0, 0 );
    SendMessage( recordDialog->textBox, EM_LIMITTEXT, 0, 0 );

    // Save distance between text box and client area for each side.
    RECT clientRect, textBoxRect ;
    GetClientRect( hwnd, &clientRect );
    GetWindowRect( recordDialog->textBox, &textBoxRect );
    MapWindowPoints( NULL, hwnd, (LPPOINT)&textBoxRect, 2 );
    recordDialog->leftDiff   = textBoxRect.left ;
    recordDialog->topDiff    = textBoxRect.top  ;
    recordDialog->rightDiff  = clientRect.right  - textBoxRect.right  ;
    recordDialog->bottomDiff = clientRect.bottom - textBoxRect.bottom ;
    SetFocus(recordDialog->textBox);
    SendMessage( recordDialog->textBox, EM_SETSEL, (WPARAM)-1, 0 );
    return FALSE ;
  }
  return FALSE ;
}

String& RecordDialog::getWindowText( HWND hwnd, String& s ) {
  const UINT textLength = GetWindowTextLength( hwnd );
  if ( textLength != (UINT)-1 ) {
    TCHAR* buffer = (TCHAR*) malloc( sizeof(TCHAR) * (textLength+1) ) ;
    GetWindowText( hwnd, buffer, textLength+1 );
    s = buffer ;
    free( buffer );
  }
  return s ;
}

void RecordDialog::onCancel() {
  setVisible( false );
}

INT_PTR RecordDialog::onCommand( WPARAM wParam, LPARAM lParam ) {
  WORD id = LOWORD( wParam ) ;
  switch ( id ) {
    case IDCANCEL: onCancel() ; break ;
    default:                    break ;
  }
  return TRUE ;
}

INT_PTR RecordDialog::onCtlColorEdit( WPARAM wParam, LPARAM lParam ) {
  HDC hdc = (HDC) wParam ;
  SetTextColor( hdc, color );
  SetBkColor( hdc, backgroundBrush.getColor() );
  return (INT_PTR) backgroundBrush.getHandle() ;
}

INT_PTR RecordDialog::onMessage( HWND hwnd,
                                 UINT   msg,
                                 WPARAM wParam,
                                 LPARAM lParam ) {
  switch ( msg ) {
    case WM_COMMAND:      return onCommand( wParam, lParam ) ;
    case WM_CTLCOLOREDIT: return onCtlColorEdit( wParam, lParam );
    case WM_SIZE:         return onSize( wParam, lParam );
    default:              return FALSE ;
  }
}

INT_PTR RecordDialog::onSize( WPARAM wParam, LPARAM lParam ) {
  const WORD w = LOWORD( lParam );
  const WORD h = HIWORD( lParam );
  const int textBoxWidth  = w - leftDiff - rightDiff ;
  const int textBoxHeight = h - topDiff - bottomDiff ;
  SetWindowPos( textBox, NULL, leftDiff, topDiff,
                textBoxWidth, textBoxHeight, SWP_NOZORDER );
                
  RECT rcNameBox ;
  GetWindowRect( nameBox, &rcNameBox );
  MapWindowPoints( NULL, hwnd, (LPPOINT)&rcNameBox, 2 );
  const int leftMargin = rcNameBox.left ;
  const int topMargin = rcNameBox.top ;
  const int nameBoxWidth = w - leftMargin - rightDiff ;
  const int nameBoxHeight = rcNameBox.bottom - rcNameBox.top ;
  SetWindowPos( nameBox, NULL, leftMargin, topMargin,
                nameBoxWidth, nameBoxHeight, SWP_NOZORDER );
                
  return TRUE ;
}
