#include "Window.h"
#include "CustomWindow.h"
#include <cstdio>
#include "ListBox.h"
#include "Splitter.h"
#include "Menu.h"
#include "TextBox.h"
#include "Button.h"
#include "ColorDialog.h"
#include "Font.h"
#include "FileDialog.h"
#include "Brush.h"
#include "InputDialog.h"
#include "WinString.h"
#include "AcceleratorTable.h"
#include "Notebook.h"
#include "Settings.h"
#include "ConvertUtils.h"
#include "WinUtils.h"

using namespace std;

class MainWindow : public CustomWindow, Splitter::Listener {
public:
  enum MenuId {
    MI_NEW=1000, MI_OPEN, MI_SAVE, MI_SAVE_AS, MI_EXIT,
    MI_ADD,MI_RENAME,MI_REMOVE,MI_REMOVE_ALL,MI_SORT_ASC,MI_SORT_DES,
    MI_ABOUT, MI_FONT, MI_COLOR, MI_BACKGROUND_COLOR, MI_SELECT_ALL
  };
  static const int SEARCHBOX_HEIGHT = 20 ;
  static const int LISTBOX_TOP = 2 + SEARCHBOX_HEIGHT + 2 ;
  static const int SEARCHBUTTON_WIDTH = 60 ;
  static const TCHAR WINDOW_TITLE[] ;
  static const TCHAR ABOUT_TITLE[] ;
  static const TCHAR ABOUT[] ;
  static const TCHAR SETTINGS_FILENAME[] ;
  static const int RECENT_FILES_MENU_ID_START = 12345 ;
  //static const int RECENT_FILES_MAX = 10 ;
public:
  MainWindow();
  void addRecentFile( String filename );
  bool confirmBox( const String& );
  void clearRecentFilesMenu();
  void displayRecordsPopupMenu(int screenX, int screenY);
  void displaySelectedRecord();
  String findSettingsFilePath();
  void initAcceleratorTable();
  void initMenus();
  bool isOnRightClickListBox( const MSG& msg ) const ;
  bool isReturnKeyPressedInSearchBox( const MSG& msg ) const ;
  bool loadFile( const String& fileToLoad );
  bool loadSettings();
  void onAbout();
  void onAdd();
  void onBackgroundColor();
  LRESULT onClose( WPARAM, LPARAM );
  void onColor();
  LRESULT onCtlColorEdit( WPARAM, LPARAM );
  LRESULT onCtlColorListBox( WPARAM, LPARAM );
  LRESULT onCommand( WPARAM, LPARAM );
  LRESULT onDropFiles( WPARAM, LPARAM );
  void onExit();
  void onFont();
  LRESULT onInitMenuPopup( WPARAM wParam, LPARAM lParam );
  void onItemSelectionChange();
  virtual LRESULT onMessage( UINT msg, WPARAM wParam, LPARAM lParam );
  LRESULT onMove( WPARAM wParam, LPARAM lParam );
  virtual void onMove( Splitter& splitter ) ;
  void onNew();
  void onOpen() ;
  void onRemove() ;
  void onRemoveAll() ;
  void onRename() ;
  void onRightClickListBox();
  void onSave() ;
  void onSaveAs() ;
  void onSearch();
  void onSelectAll();
  LRESULT onSize( WPARAM, LPARAM );
  void removeInvalidRecentFiles();
  int run( HINSTANCE hInst, HINSTANCE hPrev, LPSTR args, int nShow );
  bool saveSettings();
  void updateDisplayedRecord();
private:
  ListBox          listBox;
  Splitter         splitter ;
  Menu             menuBar, fileMenu, recordsMenu, helpMenu, viewMenu,
                   recentFilesMenu ;
  TextBox          nameBox, valueBox, searchBox ;
  Button           searchButton ;
  ColorDialog      colorDialog ;
  Font             font ;
  Notebook         notebook ;
  InputDialog      inputDialog ;
  AcceleratorTable acceleratorTable ;
  String           fileName ;
  String           displayedRecordName ;
  Settings         settings ;
private:
  static MainWindow* mainWindow ;
};

//------------------------------------------------------------------------------
// Application Static Data
//------------------------------------------------------------------------------

#define APP_NAME    _T( "Andrew's Notebook" )
#define APP_VERSION _T( "9.6.16" )
#define APP_TITLE   APP_NAME _T(" ") APP_VERSION
const TCHAR MainWindow::WINDOW_TITLE[] = APP_TITLE _T(" - Untitled");
const TCHAR MainWindow::ABOUT_TITLE[] = _T("About " APP_NAME);
const TCHAR MainWindow::ABOUT[] =
  APP_NAME _T(" - A Simple Note Taking Tool\r\n"
  "Version " APP_VERSION "\r\n"
  "Copyright 2009 Andrew Lim\r\n"
  "danteshamest@gmail.com\r\n\r\n"
  "Created with Code::Blocks and UPX\r\n"
);
const TCHAR MainWindow::SETTINGS_FILENAME[] = _T("settings.txt");

MainWindow* MainWindow::mainWindow = NULL ;

MainWindow::MainWindow()
  : CustomWindow( WINDOW_TITLE, NULL, WS_OVERLAPPEDWINDOW, WS_EX_ACCEPTFILES )
  , listBox( *this, ListBox::DEFAULT_STYLE|WS_CLIPSIBLINGS|LBS_SORT,
             WS_EX_CLIENTEDGE)
  , splitter( *this )
  , menuBar( true )
  , nameBox( *this, WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|ES_READONLY,
             WS_EX_CLIENTEDGE )
  , valueBox( *this, TextBox::NOTEPAD_STYLE|WS_CLIPSIBLINGS, WS_EX_CLIENTEDGE )
  , searchBox( *this, WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS, WS_EX_CLIENTEDGE )
  , searchButton( *this, _T("Search"), WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS )
  , inputDialog(*this,_T("New note"),WS_OVERLAPPEDWINDOW,WS_EX_TOOLWINDOW)
{
  HICON hIconMain = LoadIcon( GetModuleHandle(NULL), _T("ID_MAINICON") );
  setIcon( ICON_BIG, hIconMain );
  setIcon( ICON_SMALL, hIconMain );

  mainWindow = this ;
  initAcceleratorTable();
  initMenus(); // put initMenus() before loadSettings(), otherwise window pos screws up
  loadSettings() ;
  splitter.setSplitterListener( *this );
  splitter.setSize( 4, 300 );
  splitter.setLeft( settings.splitterX );

  listBox.setHorizontalExtent( 800 );
  listBox.setTop( LISTBOX_TOP );
  splitter.setTop( LISTBOX_TOP );
  nameBox.setTop(  LISTBOX_TOP );

  nameBox.setReadOnly( true );
  font.create( settings.fontName.c_str(), settings.fontSize, settings.fontBold,
               settings.fontItalic );
  HFONT hFont = font.getHandle();
  nameBox.setFont( hFont );
  valueBox.setFont( hFont );
  searchBox.setFont( hFont );
  listBox.setFont( hFont );
  searchButton.setDefaultGuiFont();

  displaySelectedRecord() ;

  bool windowMaximizedCache = settings.windowMaximized ;
  SetWindowPos( *this, NULL, settings.windowX, settings.windowY,
                settings.windowWidth, settings.windowHeight,
                SWP_NOZORDER );
  if ( windowMaximizedCache ) {
    ShowWindow( *this, SW_SHOWMAXIMIZED );
  }
  else {
    ShowWindow( *this, SW_SHOW );
  }
}

void MainWindow::addRecentFile( String filename ) {
  std::deque<String>& recentFiles = settings.recentFiles ;
  std::deque<String>::iterator i = std::find( recentFiles.begin(),
                                              recentFiles.end(),
                                              filename );

  // Remove the filename before adding it - we do not want duplicates
  if ( i != recentFiles.end() ) {
    recentFiles.erase( i );
  }
  recentFiles.push_front( filename );

  // Do not exceed the maximum size
  if ( (int)recentFiles.size() > settings.recentFilesMax ) {
    recentFiles.pop_back();
  }
}

bool MainWindow::confirmBox( const String& msg ) {
  return messageBox( msg.c_str(), _T("Confirm"), MB_YESNO ) == IDYES ;
}

void MainWindow::clearRecentFilesMenu() {
  int count = recentFilesMenu.itemCount() ;
  for ( int i=count-1; i>=0; --i ) {
    recentFilesMenu.removeByPosition( i );
  }
}

void MainWindow::displayRecordsPopupMenu( int screenX, int screenY ) {
  TrackPopupMenu( recordsMenu, TPM_LEFTALIGN|TPM_TOPALIGN, screenX, screenY, 0,
                  *this, 0 );
}

String MainWindow::findSettingsFilePath() {
  String settingsFilePath( WinUtils::getProgramDirectory() );
  settingsFilePath += _T("\\") ;
  settingsFilePath += SETTINGS_FILENAME ;
  return settingsFilePath ;
}

/**
 * Displays the record currently selected in the list box.
 * If no record is currently selected, the name and value textboxes will be
 * cleared and made readonly.
 */
void MainWindow::displaySelectedRecord() {
  // Get the index of the record selected in the list box.
  UINT selectedIndex = listBox.getSelectedIndex() ;
  if ( selectedIndex != (UINT) -1 ) {
    // get the key
    String key ; listBox.getItemText( selectedIndex, key );
    // get the value
    String value( notebook.get(key.c_str())->value );
    // display key and value in their respective textboxes
    valueBox.setReadOnly( false );
    nameBox.setText( key.c_str() );
    valueBox.setText( value.c_str() );

    displayedRecordName = key ;
  }
  else {
    nameBox.setText( _T("") );
    valueBox.setText( _T("") );
    valueBox.setReadOnly( true );

    displayedRecordName = _T("") ;
  }
  // prevent strange flicker around value box when mouseover
  RECT rc = valueBox.getRectRelative( *this ) ;
  invalidateRect( &rc, FALSE );

}

void MainWindow::initAcceleratorTable() {
  static ACCEL accels[] = {
      { FCONTROL|FVIRTKEY, 'S', MI_SAVE }       // Ctrl+S
    , { FCONTROL|FVIRTKEY, 'O', MI_OPEN }       // Ctrl+O
    , { FCONTROL|FVIRTKEY, 'N', MI_NEW }        // Ctrl+N
    , { FCONTROL|FVIRTKEY, 'A', MI_SELECT_ALL } // Ctrl+A
    , { FCONTROL|FVIRTKEY, 'R', MI_RENAME }        // Ctrl+R
    , { FCONTROL|FVIRTKEY, 0x000000bb, MI_ADD } // Ctrl++
    , { FCONTROL|FVIRTKEY, 0x0000006b, MI_ADD } // Ctrl++
    , { FCONTROL|FVIRTKEY, VK_DELETE, MI_REMOVE } // Ctrl+Del
  };

  const int ACCEL_COUNT = sizeof( accels ) / sizeof( accels[0] ) ;
  if ( !acceleratorTable.create( accels, ACCEL_COUNT ) ) {
    MessageBox( *this, _T("Error creating accelerator table!"), 0, 0 );
  }
}

void MainWindow::initMenus() {
  fileMenu.add( _T("New\tCtrl+N"), MI_NEW );
  fileMenu.add( _T("Open...\tCtrl+O"), MI_OPEN );
  fileMenu.add( recentFilesMenu, _T("&Recent Files") );
  fileMenu.add( _T("Save\tCtrl+S"), MI_SAVE );
  fileMenu.add( _T("Save As..."), MI_SAVE_AS );
  fileMenu.addSeparator();
  fileMenu.add( _T("Exit\tEsc"), MI_EXIT );

  recordsMenu.add( _T("Add...\tCtrl++"), MI_ADD );
  recordsMenu.add( _T("Rename...\tCtrl+R"), MI_RENAME );
  recordsMenu.add( _T("Remove...\tCtrl+Del"), MI_REMOVE );
  recordsMenu.add( _T("Remove All..."), MI_REMOVE_ALL );

  helpMenu.add( _T("About ") APP_NAME, MI_ABOUT );

  viewMenu.add( _T("&Font..."), MI_FONT );
  viewMenu.add( _T("&Color..."), MI_COLOR );
  viewMenu.add( _T("&Background Color..."), MI_BACKGROUND_COLOR );

  menuBar.add( fileMenu, _T("&File") );
  menuBar.add( recordsMenu, _T("&Notes") );
  menuBar.add( viewMenu, _T("&View") );
  menuBar.add( helpMenu, _T("&Help") );

  menuBar.attachToWindow( *this );
}

bool MainWindow::isOnRightClickListBox( const MSG& msg ) const {
  return msg.message == WM_RBUTTONDOWN && msg.hwnd == listBox ;
}

bool MainWindow::isReturnKeyPressedInSearchBox( const MSG& msg ) const {
  return msg.hwnd==searchBox&&msg.message==WM_KEYDOWN&&msg.wParam==VK_RETURN ;
}

bool MainWindow::loadFile( const String& fileToLoad ) {
  if ( notebook.openFile( fileToLoad.c_str() ) ) {
    fileName = fileToLoad ;
    String title( APP_TITLE _T(" - ") + fileName );
    setText( title.c_str() );
    listBox.removeAll() ;
    std::map<String,Note> notes = notebook.notes ;
    for ( std::map<String,Note>::iterator i=notes.begin();
          i != notes.end(); ++i ) {
      const String& name = i->first ;
      listBox.add( name.c_str() );
    }
    displaySelectedRecord();
    return true ;
  }
  return false ;
}

bool MainWindow::loadSettings() {
  String absoluteSettingsPath = findSettingsFilePath();
  return settings.load( absoluteSettingsPath );
}

void MainWindow::onAbout() {
  MessageBox( *this, ABOUT, ABOUT_TITLE, MB_ICONINFORMATION );
}

void MainWindow::onAdd() {
  inputDialog.centerInScreen();
  inputDialog.getTextBox().setText( _T("") );
  if ( inputDialog.show() ) {
    String newName ;
    inputDialog.getInputString( newName );
    if ( !newName.empty() ) {
      if ( !notebook.get( newName.c_str() ) ) {
        listBox.add( newName.c_str() );
        notebook.put( Note(newName) );

        // select the string
        UINT index = listBox.findStringExact( -1, newName.c_str() );
        if ( index != (UINT)LB_ERR ) {
          updateDisplayedRecord();
          listBox.setSelectedIndex( index );
          displaySelectedRecord();
          valueBox.setFocus();
        }
      }
      else {
        messageBox( _T("A note with that name already exists."),
                    _T("Oops!"), MB_OK );
      }
    }
    else {
      messageBox( _T("Note name cannot be empty."), _T("Oops!"), MB_OK );
    }
  }
}

void MainWindow::onBackgroundColor() {
  if ( colorDialog.show( *this ) ) {
    settings.backgroundColorBrush.createSolid( colorDialog.getColor() );

    // Redraw controls
    invalidateRect( NULL, FALSE );
    listBox.invalidateRect( NULL, TRUE );
    nameBox.invalidateRect( NULL, TRUE );
    valueBox.invalidateRect( NULL, TRUE );
    searchBox.invalidateRect( NULL, TRUE );
  }
}

LRESULT MainWindow::onClose( WPARAM wParam, LPARAM lParam ) {
  PostQuitMessage( 0 );
  return 0 ;
}

void MainWindow::onColor() {
  if ( colorDialog.show( *this ) ) {
    // Set new color
    settings.colorBrush.createSolid( colorDialog.getColor() );

    // Redraw controls
    listBox.invalidateRect( NULL, TRUE );
    nameBox.invalidateRect( NULL, TRUE );
    valueBox.invalidateRect( NULL, TRUE );
    searchBox.invalidateRect( NULL, TRUE );
    invalidateRect( NULL, TRUE );
  }
}

LRESULT MainWindow::onCommand( WPARAM wParam, LPARAM lParam ) {
  int  id   = LOWORD( wParam ) ;
  int  code = HIWORD( wParam ) ;
  HWND ctrl = (HWND) lParam ;
  if ( ctrl == NULL && code == 0 ) { // menu
    if ( id>=RECENT_FILES_MENU_ID_START &&
         id<RECENT_FILES_MENU_ID_START+(int)settings.recentFiles.size() ) {
      int menuItemIndex = id-RECENT_FILES_MENU_ID_START ;
      String fileToLoad = settings.recentFiles[ menuItemIndex ];
      if ( !loadFile( fileToLoad ) ) {
        String msg( String(_T("Error loading file: ")) + fileToLoad );
        messageBox( msg.c_str(), 0, 0);
      } else {
        addRecentFile( fileName );
      }
    }
  }
  else if ( ctrl == listBox ) {
    switch ( code ) {
      case LBN_SELCHANGE: onItemSelectionChange();   return 0 ;
      case LBN_SELCANCEL: onItemSelectionChange();   return 0 ;
      default:                                       return 0 ;
    }
  }
  else if ( ctrl == searchButton ) {
    onSearch();
    return 0 ;
  }
  switch ( id ) {
    case MI_ABOUT:            onAbout();           return 0 ;
    case MI_ADD:              onAdd();             return 0 ;
    case MI_BACKGROUND_COLOR: onBackgroundColor(); return 0 ;
    case MI_COLOR:            onColor();           return 0 ;
    case MI_FONT:             onFont();            return 0 ;
    case MI_NEW:              onNew();             return 0 ;
    case MI_OPEN:             onOpen();            return 0 ;
    case MI_REMOVE:           onRemove();          return 0 ;
    case MI_REMOVE_ALL:       onRemoveAll();       return 0 ;
    case MI_RENAME:           onRename();          return 0 ;
    case MI_SAVE:             onSave();            return 0 ;
    case MI_SAVE_AS:          onSaveAs();          return 0 ;
    case MI_SELECT_ALL:       onSelectAll();       return 0 ;
    case MI_EXIT:             onExit();            return 0 ;
    default:                                       return 0 ;
  }
}

LRESULT MainWindow::onCtlColorEdit( WPARAM wParam, LPARAM lParam ) {
  HDC hdc = (HDC) wParam ;
  SetTextColor( hdc, settings.colorBrush.getColor() );
  SetBkColor( hdc, settings.backgroundColorBrush.getColor() );
  return (LRESULT) settings.backgroundColorBrush.getHandle() ;
}

LRESULT MainWindow::onCtlColorListBox( WPARAM wParam, LPARAM lParam ) {
  HDC hdc = (HDC) wParam ;
  SetTextColor( hdc, settings.colorBrush.getColor() );
  SetBkColor( hdc, settings.backgroundColorBrush.getColor() );
  return (LRESULT) settings.backgroundColorBrush.getHandle() ;
}

LRESULT MainWindow::onDropFiles( WPARAM wParam, LPARAM lParam ) {
  HDROP hDrop = (HDROP) wParam ;
  TCHAR filePath[ MAX_PATH ];
  if ( DragQueryFile( hDrop, 0, filePath, MAX_PATH ) ) {
    if ( !loadFile( filePath ) ) {
      String msg( String(_T("Error loading file: ")) + filePath );
      messageBox( msg.c_str(), 0, 0);
    } else {
      addRecentFile( fileName );
    }
  }
  return 0 ;
}

void MainWindow::onExit() {
  PostQuitMessage( 0 );
}

void MainWindow::onFont() {
  if ( font.showFontDialog( getHandle() ) ) {
    const HFONT hFont = font.getHandle() ;
    if ( hFont ) {
      listBox.setFont( hFont );
      nameBox.setFont( hFont );
      valueBox.setFont( hFont );
      searchBox.setFont( hFont );

      settings.fontName = font.getFontName();
      settings.fontSize = font.getHeight();
      settings.fontBold = font.isBold();
      settings.fontItalic = font.isItalic();
    }
  }
}

void MainWindow::onItemSelectionChange() {
  UINT selectedIndex = listBox.getSelectedIndex() ;
  if ( selectedIndex != (UINT) -1 ) {
    String key ;
    listBox.getItemText( selectedIndex, key );
    if ( displayedRecordName != key ) {
      updateDisplayedRecord();
      displaySelectedRecord();
    }
  }
}

LRESULT MainWindow::onInitMenuPopup( WPARAM wParam, LPARAM lParam ) {
  HMENU initMenu = (HMENU) wParam ;
  HMENU hMenu = getMenu();
  bool selected = listBox.getSelectedIndex() != (UINT) -1 ;
  EnableMenuItem( hMenu, MI_RENAME, selected?MF_ENABLED:MF_GRAYED) ;
  EnableMenuItem( hMenu, MI_REMOVE, selected?MF_ENABLED:MF_GRAYED) ;
  bool exists = notebook.notes.size() > 0 ;
  EnableMenuItem( hMenu, MI_REMOVE_ALL, exists?MF_ENABLED:MF_GRAYED) ;
  if ( initMenu == recentFilesMenu ) {
    clearRecentFilesMenu();
    removeInvalidRecentFiles();
    std::deque<String>& recentFiles = settings.recentFiles ;
    for ( size_t i=0; i<recentFiles.size(); ++i ) {
      recentFilesMenu.add(recentFiles[i].c_str(), RECENT_FILES_MENU_ID_START+i);
    }
  }
  return 0 ;
}

LRESULT MainWindow::onMessage( UINT msg, WPARAM wParam, LPARAM lParam ) {
  switch ( msg ) {
    case WM_CLOSE:           return onClose( wParam, lParam );
    case WM_COMMAND:         return onCommand( wParam, lParam );
    case WM_CTLCOLOREDIT:    return onCtlColorEdit( wParam, lParam );
    case WM_CTLCOLORLISTBOX: return onCtlColorListBox( wParam, lParam );
    case WM_DROPFILES:       return onDropFiles( wParam, lParam );
    case WM_INITMENUPOPUP:   return onInitMenuPopup( wParam, lParam );
    case WM_SIZE:            return onSize( wParam, lParam );
    case WM_MOVE:            return onMove( wParam, lParam );
    default:                 return CustomWindow::onMessage(msg,wParam,lParam);
  }
}

void MainWindow::onNew() {
  if ( confirmBox(_T("Start a new project?")) ) {
    fileName.clear() ;
    setText( WINDOW_TITLE );
    notebook.notes.clear() ;
    listBox.removeAll() ;
    displaySelectedRecord();
  }
}

LRESULT MainWindow::onMove( WPARAM wParam, LPARAM lParam ) {
  const HWND hwnd = *this ;
  if ( !IsZoomed(hwnd) && !IsIconic(hwnd) ) {
    RECT rcWindow = {0};
    GetWindowRect( hwnd, &rcWindow );
    settings.windowX = rcWindow.left ;
    settings.windowY = rcWindow.top ;
  }
  return 0 ;
}

void MainWindow::onMove( Splitter& splitter ) {
  nameBox.setRedraw( false );
  valueBox.setRedraw( false );

  POINT splitterLocation = splitter.getLocation() ;

  int clientW, clientH ;
  getClientSize( clientW, clientH );

  RECT listBoxMargins ;
  listBoxMargins.top    = searchBox.getTop() + searchBox.getHeight() + 2 ;
  listBoxMargins.left   = 2 ;
  listBoxMargins.bottom = 0 ;
  listBoxMargins.right  = clientW - splitterLocation.x ;
  listBox.setMargins( &listBoxMargins );

  RECT margins = { splitterLocation.x + splitter.getWidth(), LISTBOX_TOP };
  nameBox.setMargins( &margins, Window::LEFT|Window::TOP|Window::RIGHT );
  nameBox.setHeight( 22 );

  RECT valueBoxMargins = { margins.left,
                           nameBox.getTop() + nameBox.getHeight() + 2 };
  valueBox.setMargins( &valueBoxMargins, Window::ALL );


  nameBox.setRedraw( true );
  valueBox.setRedraw( true );

  nameBox.invalidateRect( NULL, FALSE );
  valueBox.invalidateRect( NULL, FALSE );
}

void MainWindow::onOpen() {
  OpenFileDialog dialog ;
  if ( dialog.showDialog( *this ) ) {
    if ( !loadFile( dialog.getFileName() ) ) {
      String msg( String(_T("Error loading file: ")) + dialog.getFileName() );
      messageBox( msg.c_str(), 0, 0);
    } else {
      addRecentFile( fileName );
    }
  }
}

void MainWindow::onRemove() {
  UINT selectedIndex = listBox.getSelectedIndex() ;
  if ( selectedIndex != (UINT) -1 ) {
    String key ;
    listBox.getItemText( selectedIndex, key );
    String msg = String( _T("Remove \"") ) + key + _T("\"?") ;
    if ( confirmBox(msg)  ) {
      notebook.remove( key.c_str() );
      listBox.remove( selectedIndex );

      displaySelectedRecord();
    }
  }
}

void MainWindow::onRemoveAll() {
  if ( confirmBox(_T("Remove all records?")) ) {
    if ( notebook.notes.size() > 0 ) {
      notebook.notes.clear();
      listBox.removeAll();
      displaySelectedRecord();
    }
  }
}

void MainWindow::onRename() {
  UINT selectedIndex = listBox.getSelectedIndex() ;
  if ( selectedIndex != (UINT) -1 ) {
    String currentName ;
    listBox.getItemText( selectedIndex, currentName );
    inputDialog.centerInScreen();
    inputDialog.setInputString( currentName );
    inputDialog.getTextBox().selectAll();
    if ( inputDialog.show() ) {
      String newName ;
      inputDialog.getInputString( newName );
      // same name
      if ( newName == currentName ) {
        // don't do anything
      }
      // cannot be blank
      else if ( newName.empty() ) {
        messageBox(_T("Note name cannot be empty."), _T("Oops!"), MB_OK );
      }
      // new name
      else if ( !notebook.get( newName.c_str() ) ) {
        updateDisplayedRecord();

        // update map
        notebook.rename( currentName.c_str(), newName.c_str() );

        // update listbox
        listBox.remove( selectedIndex );
        int newIndex = listBox.add( newName.c_str() );
        listBox.setSelectedIndex( newIndex );

        // display new name
        nameBox.setText( newName.c_str() );
        displayedRecordName = newName ;
      }
      // existing name
      else {
        messageBox( _T("A note with that name already exists."),
                    _T("Oops!"), MB_OK );
      }
    }
  }
}

void MainWindow::onRightClickListBox() {
  POINT cursorPos;
  GetCursorPos( &cursorPos );

  // Convert screen coordinates to listbox coordinates
  POINT listBoxPos = cursorPos ;
  MapWindowPoints( NULL, listBox, &listBoxPos, 1 );

  UINT nItem = listBox.getItemFromPoint( listBoxPos.x, listBoxPos.y );

  // if right-clicked an item, select that item and display its contents
  if ( nItem != (UINT)-1) {
    updateDisplayedRecord();
    listBox.setSelectedIndex( (UINT)nItem );
    displaySelectedRecord();
  }
  // display the popup menu
  displayRecordsPopupMenu( cursorPos.x, cursorPos.y );
}

void MainWindow::onSave() {
  updateDisplayedRecord();
  if ( !fileName.empty() ) {
    if ( !notebook.saveFile( fileName.c_str() ) ) {
      MessageBox(*this, _T("Error saving file!"), 0, 0);
    } else {
      addRecentFile( fileName );
    }
  }
  else {
    onSaveAs() ;
  }
}

void MainWindow::onSaveAs() {
  updateDisplayedRecord();
  SaveFileDialog dialog ;
  if ( dialog.showDialog( *this ) ) {
    if ( notebook.saveFile( dialog.getFileName() ) ) {
      fileName = dialog.getFileName() ;
      const String title = APP_TITLE _T(" - ") + fileName ;
      setText( title.c_str() );
      addRecentFile( fileName );
    }
    else {
      messageBox( _T("Error saving file!"), 0, 0);
    }
  }
}

void MainWindow::onSearch() {
  String searchString ;
  searchBox.getText( searchString );
  if ( searchString.empty() ) {
    if ( listBox.size() != (UINT)notebook.notes.size() ) {
      updateDisplayedRecord();
      listBox.removeAll() ;
      std::map<String,Note> values = notebook.notes ;
      std::map<String,Note>::iterator i=values.begin();
      for (;i!=values.end();++i){
        const String& name = i->first ;
        listBox.add( name.c_str() );
      }
      displaySelectedRecord();
    }
  }
  else {
    updateDisplayedRecord();
    listBox.removeAll() ;
    vector<String> keys ;
    notebook.search( searchString, keys );
    for ( size_t i=0; i<keys.size(); ++i ) {
      listBox.add( keys[ i ].c_str() );
    }
    displaySelectedRecord();
  }
}

void MainWindow::onSelectAll() {
  HWND hwndFocus = GetFocus();
  if ( hwndFocus == searchBox ||
       hwndFocus == valueBox  ||
       hwndFocus == inputDialog.getTextBox().getHandle() ) {
    TextBox::selectAll( hwndFocus );
  }
}

LRESULT MainWindow::onSize( WPARAM wParam, LPARAM lParam ) {
  int clientHeight = HIWORD( lParam );
  const int resizeFlag = (int) wParam ;

  RECT searchBoxMargins ;
  searchBoxMargins.left   = 2 ;
  searchBoxMargins.top    = 2 ;
  searchBoxMargins.right  = 2 + SEARCHBUTTON_WIDTH + 2 ;
  searchBoxMargins.bottom = clientHeight - 2 - SEARCHBOX_HEIGHT ;
  searchBox.setMargins( &searchBoxMargins );

  RECT searchButtonMargins ;
  searchButtonMargins.left = searchBox.getLeft() + searchBox.getWidth() + 2 ;
  searchButtonMargins.right = 2 ;
  searchButtonMargins.top = 2 ;
  searchButtonMargins.bottom = clientHeight - 2 - SEARCHBOX_HEIGHT ;
  searchButton.setMargins( &searchButtonMargins );

  splitter.setHeight( clientHeight );
  onMove( splitter );

  if ( resizeFlag == SIZE_MAXIMIZED ) {
    settings.windowMaximized = true ;
  }
  else if ( resizeFlag == SIZE_RESTORED ) {
    settings.windowMaximized = false ;
    RECT rcWindow ;
    GetWindowRect( *this, &rcWindow );
    settings.windowX = rcWindow.left ;
    settings.windowY = rcWindow.top ;
    settings.windowWidth = rcWindow.right - rcWindow.left ;
    settings.windowHeight = rcWindow.bottom - rcWindow.top ;
  }

  return 0 ;
}

/*
 * This function updates the value (based on user input) of the record being
 * displayed.
 */
void MainWindow::updateDisplayedRecord() {
  if ( !displayedRecordName.empty() ) {
    // get new value
    String value ; valueBox.getText(value) ;
    // update record value
    notebook.put( Note(displayedRecordName, value) );
  }
}

void MainWindow::removeInvalidRecentFiles() {
  std::deque<String>& recentFiles = settings.recentFiles ;
  for( int i=recentFiles.size()-1; i>=0; i-- ) {
    String recentFile = recentFiles[ i ];
    if ( !WinUtils::isExistingFile(recentFile) ) {
      recentFiles.erase( recentFiles.begin() + i );
      i = recentFiles.size() - 1 ;
    }
  }
}

int MainWindow::run(HINSTANCE hInst, HINSTANCE hPrev, LPSTR args, int nShow ){
  // If there is a command line argument, treat it as a file name, and try to
  // load it.
  TCHAR* cmdLine = GetCommandLine() ;
  int argc = 0 ;
  LPWSTR* argv = CommandLineToArgvW( cmdLine, &argc );
  if ( argc > 1 && _tcslen(argv[1]) ) {
    if ( !loadFile(argv[1]) ) {
      String msg( String(_T("Error loading file: ")) + argv[1] );
      messageBox( msg.c_str(), 0, 0 );
    }
  }

  MSG msg ;
  while( (GetMessage(&msg,0,0,0) > 0) ) {
    if ( isReturnKeyPressedInSearchBox(msg) ) {
      onSearch();
    }
    if ( isOnRightClickListBox(msg) ) {
      onRightClickListBox();
    }
    else if ( !acceleratorTable.translate(*this,&msg) ) {
      if ( msg.hwnd==valueBox || // TAB key works for the valuebox
           !IsDialogMessage(*this,&msg) ) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
      }
    }
  }

  saveSettings();

  return (int)msg.wParam ;
}

bool MainWindow::saveSettings() {
  settings.splitterX = splitter.getLeft();
  String absoluteSettingsPath = findSettingsFilePath();
  return settings.save( absoluteSettingsPath );
}

int WINAPI WinMain( HINSTANCE hInst, HINSTANCE hPrev, LPSTR args, int nShow ) {
  try {
    InitCommonControls();
    MainWindow window ;
    return window.run( hInst, hPrev, args, nShow );
  }
  catch ( WindowsException& ex ) {
    MessageBox( NULL, ex.getMessage(), _T("Exception caught!"),
                MB_OK|MB_ICONERROR);
    return 1 ;
  }
}
